module net.BurtonRadons.dig.common.bindingList;

private import net.BurtonRadons.dig.common.dispatcher;
private import net.BurtonRadons.dig.common.event;
private import net.BurtonRadons.dig.platform.base;

/** A set of dispatchers associated with keyboard events. */

struct BindingList
{
    /** A single binding for a BindingList. */
    struct Binding
    {
        bit focus; /**< Requires keyboard focus. */
        bit shift; /**< Requires either shift key to be pressed. */
        bit control; /**< Requires either control key to be pressed. */
        char [] keyCode; /**< The key code name. */
        Dispatcher dispatcher; /**< The dispatcher to notify on event. */

        /** Convert the binding to text, such as Control-Shift-X. */
        char [] text ()
        {
            char [] t;

            if (control)
                t ~= "Control-";
            if (shift)
                t ~= "Shift-";
            return t ~ keyCode;
        }

        /** Notify the binding of an event. */
        void notify (Event event)
        {
            dispatcher.notify (event);
        }
    }

    Binding [] list; /**< The current list of bindings. */

    /** Search for the binding for an event descriptor and return the pointer or null. */
    Binding *find (bit focus, bit shift, bit control, char [] keyCode)
    {
        for (int c; c < list.length; c ++)
            if (list [c].focus == focus
             && list [c].shift == shift
             && list [c].control == control
             && !std.string.icmp (list [c].keyCode, keyCode))
                return &list [c];

        return null;
    }

    /** Search for the binding for an event and return the pointer or null. */
    Binding *find (Event event, bit focus)
    {
        return find (focus, event.shift, event.control, event.keyCode);
    }

    /** Search for the binding for a code and return the pointer or null. */
    Binding *find (char [] code)
    {
        int focus, shift, control;
        char [] keyCode;

        bindingFromCode (code, focus, shift, control, keyCode);
        return find ((bit) focus, (bit) shift, (bit) control, keyCode);
    }

    /** Dispatch an event using the binding and return whether any matches were found. */
    bit notify (Event event, bit focus)
    {
        Binding *binding = find (event, focus);

        if (binding === null)
            return false;
        binding.notify (event);
        return true;
    }

    /** Extract binding parameters from input code. */
    void bindingFromCode (char [] code, out int focus, out int shift, out int control, out char [] keyCode)
    {
        int minimum = 0;
        
        for (int c; c < code.length - 5; c ++)
            if (!icmp (code [c .. c + 5], "Focus"))
            {
                minimum = imax (c + 5, minimum);
                focus = true;
            }
        
        for (int c; c < code.length - 5; c ++)
            if (!icmp (code [c .. c + 5], "Shift"))
            {
                minimum = imax (c + 5, minimum);
                shift = true;
            }

        for (int c; c < code.length - 7; c ++)
            if (!icmp (code [c .. c + 7], "Control"))
            {
                minimum = imax (c + 7, minimum);
                control = true;
            }

        for (int c; c < code.length - 4; c ++)
            if (!icmp (code [c .. c + 4], "Ctrl"))
            {
                minimum = imax (c + 4, minimum);
                control = true;
            }

        keyCode = code;
        for (int c = code.length; c > minimum; c --)
            if (!isalpha (code [c - 1]))
            {
                keyCode = code [c .. code.length];
                break;
            }
    }

    /** Create a binding and return the dispatcher.  The code can be the keycode name
      * (such as X) along with prefixed shift, control, or ctrl in either order with any
      * separator.  For example, "Shift-CONTROL-X", or "control 4" or "CtrlLeft".  The
      * pointer is valid until the next call to bind only.
      */

    Dispatcher *bind (char [] code)
    {
        int focus, shift, control;
        char [] keyCode;

        bindingFromCode (code, focus, shift, control, keyCode);

        Binding *found;

        if ((found = find ((bit) focus, (bit) shift, (bit) control, keyCode)) !== null)
            return &found.dispatcher;

        Binding binding;

        binding.focus = (bit) focus;
        binding.shift = (bit) shift;
        binding.control = (bit) control;
        binding.keyCode = keyCode.dup;
        list ~= binding;
        return &list [list.length - 1].dispatcher;
    }

    /** Bind an event and return the dispatcher. */
    Dispatcher *bind (char [] code, Dispatcher.Method method)
    {
        Dispatcher *d = bind (code);

        d.add (method);
        return d;
    }

    /** Bind an event and return the dispatcher. */
    Dispatcher *bind (char [] code, Dispatcher.MethodB method)
    {
        Dispatcher *d = bind (code);

        d.add (method);
        return d;
    }

    /** Bind a dispatcher pointer and return the dispatcher. */
    Dispatcher *bind (char [] code, Dispatcher *dispatcher)
    {
        Dispatcher *d = bind (code);

        d.add (dispatcher);
        return d;
    }

    /** Remove a binding from the list. */
    void remove (char [] code, Dispatcher *dispatcher)
    {
        Binding *binding = find (code);

        if (binding === null)
            return;
        binding.dispatcher.remove (dispatcher);
    }
}
